from typing import Dict, Literal
import math
Point = Dict[Literal["x","y"], float]
Direction = Literal["up", "down", "left", "right", "no direction"]

def get_direction(point1: Point, point2: Point) -> Direction:
    """
    Util function to get the direction from start point to destination point.

    Args:
        point1(Point), the start point.
        point2(Point), the end point.

    Return:
        str: the direction.
    """
    # Get the coordinate of two points.
    try:
        x1, y1 = point1["x"], point1["y"]
        x2, y2 = point2["x"], point2["y"]

        assert x1 is not None
        assert x2 is not None
        assert y1 is not None
        assert y2 is not None

        vector = (x2 - x1, y2 - y1)
        vx, vy = vector
    except Exception as e:
        return "no direction"

    # Define the direction vector.
    directions = {
        "up": (0, -1),
        "down": (0, 1),
        "left": (-1, 0),
        "right": (1, 0)
    }

    vector_length = math.sqrt(vx ** 2 + vy ** 2)
    if vector_length == 0:  # same point.
        return "no direction"
    unit_vector = (vx / vector_length, vy / vector_length)

    # Calculate the cosine of each direction.
    max_cosine = -float('inf')
    closest_direction = None
    for direction, dir_vector in directions.items():
        dx, dy = dir_vector
        dir_length = math.sqrt(dx ** 2 + dy ** 2)
        cos_theta = (unit_vector[0] * dx + unit_vector[1] * dy) / dir_length
        if cos_theta > max_cosine:
            max_cosine = cos_theta
            closest_direction = direction

    return closest_direction
